import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { trpc } from "@/lib/trpc";
import { ArrowLeft, Calendar, CheckCircle2, AlertCircle } from "lucide-react";
import { Link } from "wouter";
import { useState } from "react";

export default function GoogleCalendarIntegration() {
  const { isAuthenticated } = useAuth();
  const [isConnecting, setIsConnecting] = useState(false);

  const statusQuery = trpc.googleCalendar.getStatus.useQuery(undefined, {
    enabled: isAuthenticated,
  });

  const handleConnect = () => {
    setIsConnecting(true);
    // In a real implementation, this would redirect to Google OAuth
    // For now, we'll show a message
    alert(
      "Para integração com Google Calendar, você precisará configurar as credenciais OAuth do Google no servidor."
    );
    setIsConnecting(false);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 p-4 md:p-8">
      <div className="max-w-2xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <Link href="/settings">
            <Button variant="ghost" className="mb-4">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Voltar
            </Button>
          </Link>
          <h1 className="text-3xl font-bold text-gray-900">Google Calendar</h1>
          <p className="text-gray-600 mt-2">
            Sincronize seus medicamentos com o Google Calendar
          </p>
        </div>

        {/* Status Card */}
        <Card className="border-0 shadow-lg mb-6">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Calendar className="w-5 h-5" />
              Status da Integração
            </CardTitle>
          </CardHeader>
          <CardContent>
            {statusQuery.isLoading ? (
              <div className="text-center py-8 text-gray-500">
                Verificando status...
              </div>
            ) : statusQuery.data?.connected ? (
              <div className="flex items-center gap-3 p-4 bg-green-50 rounded-lg border border-green-200">
                <CheckCircle2 className="w-6 h-6 text-green-600 flex-shrink-0" />
                <div>
                  <p className="font-semibold text-green-900">Conectado</p>
                  <p className="text-sm text-green-700">
                    Seus medicamentos estão sendo sincronizados com o Google Calendar
                  </p>
                </div>
              </div>
            ) : (
              <div className="flex items-center gap-3 p-4 bg-yellow-50 rounded-lg border border-yellow-200">
                <AlertCircle className="w-6 h-6 text-yellow-600 flex-shrink-0" />
                <div>
                  <p className="font-semibold text-yellow-900">Não Conectado</p>
                  <p className="text-sm text-yellow-700">
                    Clique no botão abaixo para conectar sua conta do Google
                  </p>
                </div>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Features */}
        <Card className="border-0 shadow-lg mb-6">
          <CardHeader>
            <CardTitle>Benefícios da Integração</CardTitle>
            <CardDescription>
              O que você pode fazer com a integração
            </CardDescription>
          </CardHeader>
          <CardContent>
            <ul className="space-y-3">
              <li className="flex items-start gap-3">
                <CheckCircle2 className="w-5 h-5 text-green-600 mt-0.5 flex-shrink-0" />
                <div>
                  <p className="font-semibold text-gray-900">
                    Sincronização Automática
                  </p>
                  <p className="text-sm text-gray-600">
                    Seus medicamentos aparecem automaticamente no seu calendário
                  </p>
                </div>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle2 className="w-5 h-5 text-green-600 mt-0.5 flex-shrink-0" />
                <div>
                  <p className="font-semibold text-gray-900">
                    Lembretes Integrados
                  </p>
                  <p className="text-sm text-gray-600">
                    Receba notificações do Google Calendar também
                  </p>
                </div>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle2 className="w-5 h-5 text-green-600 mt-0.5 flex-shrink-0" />
                <div>
                  <p className="font-semibold text-gray-900">
                    Visualização Unificada
                  </p>
                  <p className="text-sm text-gray-600">
                    Veja todos os seus compromissos e medicamentos em um só lugar
                  </p>
                </div>
              </li>
            </ul>
          </CardContent>
        </Card>

        {/* Action Buttons */}
        <div className="space-y-3">
          {!statusQuery.data?.connected && (
            <Button
              onClick={handleConnect}
              disabled={isConnecting}
              size="lg"
              className="w-full bg-blue-600 hover:bg-blue-700"
            >
              {isConnecting ? "Conectando..." : "Conectar Google Calendar"}
            </Button>
          )}

          {statusQuery.data?.connected && (
            <Button
              variant="destructive"
              size="lg"
              className="w-full"
              disabled={statusQuery.isLoading}
            >
              Desconectar Google Calendar
            </Button>
          )}

          <Button
            variant="outline"
            size="lg"
            className="w-full"
            onClick={() => window.open("https://calendar.google.com", "_blank")}
          >
            Abrir Google Calendar
          </Button>
        </div>

        {/* Info */}
        <Card className="border-0 shadow-lg mt-6 bg-blue-50">
          <CardContent className="pt-6">
            <p className="text-sm text-gray-700">
              <strong>Nota:</strong> A integração com Google Calendar requer que você
              tenha uma conta do Google. Seus dados de medicamentos serão sincronizados
              com segurança usando OAuth 2.0.
            </p>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
