import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { trpc } from "@/lib/trpc";
import { getLoginUrl } from "@/const";
import { CheckCircle2, Clock, Pill, Plus, AlertCircle, Calendar, Settings as SettingsIcon } from "lucide-react";
import { useState, useEffect } from "react";
import { useServiceWorker } from "@/hooks/useServiceWorker";
import { useMedicationNotifications } from "@/hooks/useMedicationNotifications";
import { Link } from "wouter";

export default function Home() {
  const { user, loading, isAuthenticated } = useAuth();
  useServiceWorker();
  useMedicationNotifications();
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [isAddingMed, setIsAddingMed] = useState(false);
  const [formData, setFormData] = useState({
    name: "",
    dosage: "",
    frequency: 1,
    startDate: new Date().toISOString().split("T")[0],
    endDate: "",
    color: "#3b82f6",
    schedules: [{ hour: 8, minute: 0, dayOfWeek: undefined }],
  });

  const medicationsQuery = trpc.medications.list.useQuery(undefined, {
    enabled: isAuthenticated,
  });

  const intakesQuery = trpc.intakes.today.useQuery(
    { date: selectedDate },
    { enabled: isAuthenticated }
  );

  const createMedicationMutation = trpc.medications.create.useMutation({
    onSuccess: () => {
      medicationsQuery.refetch();
      setIsAddingMed(false);
      setFormData({
        name: "",
        dosage: "",
        frequency: 1,
        startDate: new Date().toISOString().split("T")[0],
        endDate: "",
        color: "#3b82f6",
        schedules: [{ hour: 8, minute: 0, dayOfWeek: undefined }],
      });
    },
  });

  const markTakenMutation = trpc.intakes.markTaken.useMutation({
    onSuccess: () => {
      intakesQuery.refetch();
    },
  });

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100">
        <div className="text-center">
          <Pill className="w-16 h-16 mx-auto mb-4 text-blue-600 animate-bounce" />
          <p className="text-gray-600">Carregando...</p>
        </div>
      </div>
    );
  }

  if (!isAuthenticated) {
    return (
      <div className="min-h-screen flex flex-col items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100 px-4">
        <div className="text-center max-w-md">
          <div className="mb-6">
            <Pill className="w-20 h-20 mx-auto text-blue-600 mb-4" />
            <h1 className="text-4xl font-bold text-gray-900 mb-2">MedSchedule Pro</h1>
            <p className="text-gray-600">Controle seus medicamentos com facilidade</p>
          </div>

          <Card className="mb-6 border-0 shadow-lg">
            <CardContent className="pt-6">
              <p className="text-gray-700 mb-4">
                Organize seus medicamentos, receba lembretes automáticos e acompanhe sua aderência ao tratamento.
              </p>
              <ul className="text-left space-y-2 mb-6 text-sm text-gray-600">
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-4 h-4 text-green-600" />
                  Lembretes automáticos
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-4 h-4 text-green-600" />
                  Funciona offline
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-4 h-4 text-green-600" />
                  Integração com Google Calendar
                </li>
              </ul>
            </CardContent>
          </Card>

          <Button
            onClick={() => window.location.href = getLoginUrl()}
            size="lg"
            className="w-full bg-blue-600 hover:bg-blue-700"
          >
            Entrar com Manus
          </Button>
        </div>
      </div>
    );
  }

  const today = new Date();
  const isToday = selectedDate.toDateString() === today.toDateString();

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 p-4 md:p-8">
      <div className="max-w-6xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between mb-4">
            <div className="flex items-center gap-3">
              <Pill className="w-8 h-8 text-blue-600" />
              <h1 className="text-3xl font-bold text-gray-900">MedSchedule Pro</h1>
            </div>
            <div className="flex items-center gap-4">
              <div className="text-right">
                <p className="text-sm text-gray-600">Olá, {user?.name}</p>
              </div>
              <Link href="/settings">
                <Button size="icon" variant="ghost">
                  <SettingsIcon className="w-5 h-5" />
                </Button>
              </Link>
            </div>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-6">
            {/* Date Selector */}
            <Card className="border-0 shadow-lg">
              <CardHeader>
                <CardTitle>Medicamentos do Dia</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex gap-2 mb-4">
                  <input
                    type="date"
                    value={selectedDate.toISOString().split("T")[0]}
                    onChange={(e) => setSelectedDate(new Date(e.target.value))}
                    className="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                  {!isToday && (
                    <Button
                      variant="outline"
                      onClick={() => setSelectedDate(new Date())}
                    >
                      Hoje
                    </Button>
                  )}
                </div>

                {intakesQuery.isLoading ? (
                  <div className="text-center py-8 text-gray-500">
                    Carregando medicamentos...
                  </div>
                ) : intakesQuery.data && intakesQuery.data.length > 0 ? (
                  <div className="space-y-3">
                    {intakesQuery.data.map((intake) => (
                      <div
                        key={intake.id}
                        className="flex items-center justify-between p-4 bg-white rounded-lg border border-gray-200 hover:shadow-md transition"
                      >
                        <div className="flex-1">
                          <p className="font-semibold text-gray-900">
                            {medicationsQuery.data?.find((m) => m.id === intake.medicationId)?.name || "Medicamento"}
                          </p>
                          <p className="text-sm text-gray-600">
                            <Clock className="w-4 h-4 inline mr-1" />
                            {String(new Date(intake.scheduledTime).getHours()).padStart(2, "0")}:
                            {String(new Date(intake.scheduledTime).getMinutes()).padStart(2, "0")}
                          </p>
                        </div>
                        {intake.takenTime ? (
                          <div className="flex items-center gap-2 text-green-600">
                            <CheckCircle2 className="w-6 h-6" />
                            <span className="text-sm font-medium">Tomado</span>
                          </div>
                        ) : (
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => markTakenMutation.mutate({ intakeId: intake.id })}
                            disabled={markTakenMutation.isPending}
                          >
                            Marcar como tomado
                          </Button>
                        )}
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="text-center py-8">
                    <Pill className="w-12 h-12 mx-auto mb-3 text-gray-300" />
                    <p className="text-gray-500">Nenhum medicamento agendado para este dia</p>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* Navigation */}
            <Link href="/weekly" className="block">
              <Button size="lg" variant="outline" className="w-full">
                <Calendar className="w-4 h-4 mr-2" />
                Visao Semanal
              </Button>
            </Link>

            {/* Add Medication */}
            <Dialog open={isAddingMed} onOpenChange={setIsAddingMed}>
              <DialogTrigger asChild>
                <Button size="lg" className="w-full bg-blue-600 hover:bg-blue-700">
                  <Plus className="w-4 h-4 mr-2" />
                  Adicionar Medicamento
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-md">
                <DialogHeader>
                  <DialogTitle>Novo Medicamento</DialogTitle>
                  <DialogDescription>
                    Adicione um novo medicamento ao seu cronograma
                  </DialogDescription>
                </DialogHeader>

                <div className="space-y-4">
                  <div>
                    <Label htmlFor="name">Nome do Medicamento *</Label>
                    <Input
                      id="name"
                      value={formData.name}
                      onChange={(e) =>
                        setFormData({ ...formData, name: e.target.value })
                      }
                      placeholder="Ex: Dipirona"
                    />
                  </div>

                  <div>
                    <Label htmlFor="dosage">Dosagem</Label>
                    <Input
                      id="dosage"
                      value={formData.dosage}
                      onChange={(e) =>
                        setFormData({ ...formData, dosage: e.target.value })
                      }
                      placeholder="Ex: 500mg"
                    />
                  </div>

                  <div>
                    <Label htmlFor="frequency">Frequência (vezes por dia) *</Label>
                    <Select
                      value={String(formData.frequency)}
                      onValueChange={(val) =>
                        setFormData({ ...formData, frequency: parseInt(val) })
                      }
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {[1, 2, 3, 4, 5, 6].map((n) => (
                          <SelectItem key={n} value={String(n)}>
                            {n}x ao dia
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label htmlFor="startDate">Data de Início *</Label>
                    <Input
                      id="startDate"
                      type="date"
                      value={formData.startDate}
                      onChange={(e) =>
                        setFormData({ ...formData, startDate: e.target.value })
                      }
                    />
                  </div>

                  <div>
                    <Label htmlFor="endDate">Data de Término (opcional)</Label>
                    <Input
                      id="endDate"
                      type="date"
                      value={formData.endDate}
                      onChange={(e) =>
                        setFormData({ ...formData, endDate: e.target.value })
                      }
                    />
                  </div>

                  <div>
                    <Label htmlFor="color">Cor</Label>
                    <div className="flex gap-2">
                      {["#3b82f6", "#ef4444", "#10b981", "#f59e0b", "#8b5cf6"].map(
                        (c) => (
                          <button
                            key={c}
                            className={`w-8 h-8 rounded-full border-2 ${
                              formData.color === c
                                ? "border-gray-900"
                                : "border-gray-300"
                            }`}
                            style={{ backgroundColor: c }}
                            onClick={() =>
                              setFormData({ ...formData, color: c })
                            }
                          />
                        )
                      )}
                    </div>
                  </div>

                  <div>
                    <Label>Horários</Label>
                    <div className="space-y-2">
                      {formData.schedules.map((schedule, idx) => (
                        <div key={idx} className="flex gap-2">
                          <input
                            type="number"
                            min="0"
                            max="23"
                            value={schedule.hour}
                            onChange={(e) => {
                              const newSchedules = [...formData.schedules];
                              newSchedules[idx].hour = parseInt(e.target.value);
                              setFormData({ ...formData, schedules: newSchedules });
                            }}
                            className="w-16 px-2 py-1 border border-gray-300 rounded"
                            placeholder="HH"
                          />
                          <span className="text-gray-500">:</span>
                          <input
                            type="number"
                            min="0"
                            max="59"
                            value={schedule.minute}
                            onChange={(e) => {
                              const newSchedules = [...formData.schedules];
                              newSchedules[idx].minute = parseInt(e.target.value);
                              setFormData({ ...formData, schedules: newSchedules });
                            }}
                            className="w-16 px-2 py-1 border border-gray-300 rounded"
                            placeholder="MM"
                          />
                        </div>
                      ))}
                    </div>
                  </div>

                  <Button
                    onClick={() => {
                      createMedicationMutation.mutate({
                        name: formData.name,
                        dosage: formData.dosage || undefined,
                        frequency: formData.frequency,
                        startDate: new Date(formData.startDate),
                        endDate: formData.endDate ? new Date(formData.endDate) : undefined,
                        color: formData.color,
                        schedules: formData.schedules,
                      });
                    }}
                    disabled={!formData.name || createMedicationMutation.isPending}
                    className="w-full"
                  >
                    Adicionar Medicamento
                  </Button>
                </div>
              </DialogContent>
            </Dialog>

            {/* Medications List */}
            <Card className="border-0 shadow-lg">
              <CardHeader>
                <CardTitle className="text-lg">Seus Medicamentos</CardTitle>
              </CardHeader>
              <CardContent>
                {medicationsQuery.isLoading ? (
                  <p className="text-gray-500 text-sm">Carregando...</p>
                ) : medicationsQuery.data && medicationsQuery.data.length > 0 ? (
                  <div className="space-y-2">
                    {medicationsQuery.data.map((med) => (
                      <div
                        key={med.id}
                        className="p-3 rounded-lg border-l-4 bg-gray-50"
                        style={{ borderLeftColor: med.color || "#3b82f6" }}
                      >
                        <p className="font-semibold text-sm text-gray-900">
                          {med.name}
                        </p>
                        {med.dosage && (
                          <p className="text-xs text-gray-600">{med.dosage}</p>
                        )}
                        <p className="text-xs text-gray-500 mt-1">
                          {med.frequency}x ao dia
                        </p>
                      </div>
                    ))}
                  </div>
                ) : (
                  <p className="text-gray-500 text-sm">
                    Nenhum medicamento adicionado ainda
                  </p>
                )}
              </CardContent>
            </Card>

            {/* Quick Stats */}
            <Card className="border-0 shadow-lg bg-gradient-to-br from-blue-500 to-blue-600 text-white">
              <CardHeader>
                <CardTitle className="text-lg">Resumo</CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <div>
                  <p className="text-blue-100 text-sm">Total de Medicamentos</p>
                  <p className="text-3xl font-bold">
                    {medicationsQuery.data?.length || 0}
                  </p>
                </div>
                <div>
                  <p className="text-blue-100 text-sm">Agendados Hoje</p>
                  <p className="text-3xl font-bold">
                    {intakesQuery.data?.length || 0}
                  </p>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}
