import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useAdvancedAlarms } from "@/hooks/useAdvancedAlarms";
import { ArrowLeft, Bell, Volume2, Vibrate, Calendar } from "lucide-react";
import { Link } from "wouter";
import { useState, useEffect } from "react";

export default function Settings() {
  const { user, logout } = useAuth();
  const alarms = useAdvancedAlarms();
  const [testMode, setTestMode] = useState(false);

  const handleLogout = async () => {
    await logout();
    window.location.href = "/";
  };

  const handleTestAlarm = () => {
    setTestMode(true);
    alarms.triggerAlarm();
    setTimeout(() => {
      alarms.stopAlarm();
      setTestMode(false);
    }, 3000);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 p-4 md:p-8">
      <div className="max-w-2xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <Link href="/">
            <Button variant="ghost" className="mb-4">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Voltar
            </Button>
          </Link>
          <h1 className="text-3xl font-bold text-gray-900">Configurações</h1>
        </div>

        {/* Alarm Settings */}
        <Card className="border-0 shadow-lg mb-6">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Bell className="w-5 h-5" />
              Configurações de Alarmes
            </CardTitle>
            <CardDescription>
              Personalize como você recebe notificações de medicamentos
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">
            {/* Enable/Disable Alarms */}
            <div className="flex items-center justify-between">
              <div>
                <Label className="text-base font-semibold">Alarmes Ativados</Label>
                <p className="text-sm text-gray-600 mt-1">
                  Receber notificações de medicamentos
                </p>
              </div>
              <Switch
                checked={alarms.settings.enabled}
                onCheckedChange={(checked) =>
                  alarms.saveSettings({ enabled: checked })
                }
              />
            </div>

            {/* Sound Settings */}
            <div className="flex items-center justify-between">
              <div>
                <Label className="text-base font-semibold flex items-center gap-2">
                  <Volume2 className="w-4 h-4" />
                  Som do Alarme
                </Label>
                <p className="text-sm text-gray-600 mt-1">
                  Reproduzir som quando chegar a hora do medicamento
                </p>
              </div>
              <Switch
                checked={alarms.settings.soundEnabled}
                onCheckedChange={(checked) =>
                  alarms.saveSettings({ soundEnabled: checked })
                }
                disabled={!alarms.settings.enabled}
              />
            </div>

            {/* Vibration Settings */}
            <div className="flex items-center justify-between">
              <div>
                <Label className="text-base font-semibold flex items-center gap-2">
                  <Vibrate className="w-4 h-4" />
                  Vibração
                </Label>
                <p className="text-sm text-gray-600 mt-1">
                  Vibrar o dispositivo como aviso
                </p>
              </div>
              <Switch
                checked={alarms.settings.vibrationEnabled}
                onCheckedChange={(checked) =>
                  alarms.saveSettings({ vibrationEnabled: checked })
                }
                disabled={!alarms.settings.enabled}
              />
            </div>

            {/* Minutes Before */}
            <div>
              <Label className="text-base font-semibold mb-2 block">
                Notificar com Antecedência
              </Label>
              <Select
                value={String(alarms.settings.minutesBefore)}
                onValueChange={(val) =>
                  alarms.saveSettings({ minutesBefore: parseInt(val) })
                }
                disabled={!alarms.settings.enabled}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="0">No horário exato</SelectItem>
                  <SelectItem value="5">5 minutos antes</SelectItem>
                  <SelectItem value="10">10 minutos antes</SelectItem>
                  <SelectItem value="15">15 minutos antes</SelectItem>
                  <SelectItem value="30">30 minutos antes</SelectItem>
                </SelectContent>
              </Select>
              <p className="text-sm text-gray-600 mt-2">
                Você será notificado {alarms.settings.minutesBefore} minutos antes do horário agendado
              </p>
            </div>

            {/* Test Alarm */}
            <div className="pt-4 border-t">
              <Button
                onClick={handleTestAlarm}
                disabled={!alarms.settings.enabled || testMode}
                className="w-full"
                variant="outline"
              >
                {testMode ? "Testando alarme..." : "Testar Alarme"}
              </Button>
              <p className="text-xs text-gray-600 mt-2 text-center">
                Clique para ouvir como será o alarme
              </p>
            </div>
          </CardContent>
        </Card>

        {/* Google Calendar Integration */}
        <Card className="border-0 shadow-lg mb-6">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Calendar className="w-5 h-5" />
              Google Calendar
            </CardTitle>
            <CardDescription>
              Sincronize seus medicamentos com o Google Calendar
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Link href="/google-calendar" className="block">
              <Button className="w-full" variant="outline">
                Configurar Integração
              </Button>
            </Link>
          </CardContent>
        </Card>

        {/* Account Settings */}
        <Card className="border-0 shadow-lg mb-6">
          <CardHeader>
            <CardTitle>Conta</CardTitle>
            <CardDescription>Gerenciar sua conta</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <Label className="text-sm text-gray-600">Nome</Label>
              <p className="text-lg font-semibold text-gray-900">{user?.name}</p>
            </div>
            <div>
              <Label className="text-sm text-gray-600">Email</Label>
              <p className="text-lg font-semibold text-gray-900">{user?.email}</p>
            </div>
            <Button
              onClick={handleLogout}
              variant="destructive"
              className="w-full mt-4"
            >
              Sair da Conta
            </Button>
          </CardContent>
        </Card>

        {/* About */}
        <Card className="border-0 shadow-lg">
          <CardHeader>
            <CardTitle>Sobre</CardTitle>
          </CardHeader>
          <CardContent className="space-y-2">
            <div>
              <p className="text-sm text-gray-600">Versão</p>
              <p className="text-lg font-semibold text-gray-900">1.0.0</p>
            </div>
            <div>
              <p className="text-sm text-gray-600">Desenvolvido com</p>
              <p className="text-sm text-gray-900">React, TypeScript, Tailwind CSS</p>
            </div>
            <p className="text-xs text-gray-500 mt-4">
              MedSchedule Pro © 2025. Todos os direitos reservados.
            </p>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
