import { useAuth } from "@/_core/hooks/useAuth";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { trpc } from "@/lib/trpc";
import { CheckCircle2, Circle, ChevronLeft, ChevronRight } from "lucide-react";
import { useState, useMemo } from "react";

export default function WeeklyView() {
  const { isAuthenticated } = useAuth();
  const [weekStart, setWeekStart] = useState(() => {
    const today = new Date();
    const day = today.getDay();
    const diff = today.getDate() - day;
    return new Date(today.setDate(diff));
  });

  const medicationsQuery = trpc.medications.list.useQuery(undefined, {
    enabled: isAuthenticated,
  });

  const markTakenMutation = trpc.intakes.markTaken.useMutation({
    onSuccess: () => {
      // Refetch intakes for the week
      const days = Array.from({ length: 7 }, (_, i) => {
        const date = new Date(weekStart);
        date.setDate(date.getDate() + i);
        return date;
      });
      days.forEach((date) => {
        intakesQueries[date.toISOString().split("T")[0]]?.refetch?.();
      });
    },
  });

  // Fetch intakes for each day of the week
  const days = useMemo(() => {
    return Array.from({ length: 7 }, (_, i) => {
      const date = new Date(weekStart);
      date.setDate(date.getDate() + i);
      return date;
    });
  }, [weekStart]);

  const intakesQueries: Record<string, any> = {};
  days.forEach((date) => {
    const key = date.toISOString().split("T")[0];
    // eslint-disable-next-line react-hooks/rules-of-hooks
    intakesQueries[key] = trpc.intakes.today.useQuery(
      { date },
      { enabled: isAuthenticated }
    );
  });

  const dayNames = ["Dom", "Seg", "Ter", "Qua", "Qui", "Sex", "Sab"];
  const today = new Date();

  const previousWeek = () => {
    const newDate = new Date(weekStart);
    newDate.setDate(newDate.getDate() - 7);
    setWeekStart(newDate);
  };

  const nextWeek = () => {
    const newDate = new Date(weekStart);
    newDate.setDate(newDate.getDate() + 7);
    setWeekStart(newDate);
  };

  const goToToday = () => {
    const todayStart = new Date(today);
    const day = todayStart.getDay();
    const diff = todayStart.getDate() - day;
    setWeekStart(new Date(todayStart.setDate(diff)));
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 p-4 md:p-8">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-4">Visão Semanal</h1>
          <div className="flex items-center justify-between">
            <Button variant="outline" onClick={previousWeek}>
              <ChevronLeft className="w-4 h-4 mr-2" />
              Semana Anterior
            </Button>
            <div className="text-center">
              <p className="text-sm text-gray-600">
                {weekStart.toLocaleDateString("pt-BR", {
                  month: "long",
                  year: "numeric",
                })}
              </p>
            </div>
            <div className="flex gap-2">
              <Button variant="outline" onClick={goToToday}>
                Hoje
              </Button>
              <Button variant="outline" onClick={nextWeek}>
                Próxima Semana
                <ChevronRight className="w-4 h-4 ml-2" />
              </Button>
            </div>
          </div>
        </div>

        {/* Weekly Grid */}
        <div className="grid grid-cols-1 md:grid-cols-7 gap-3">
          {days.map((date, dayIndex) => {
            const dateStr = date.toISOString().split("T")[0];
            const intakes = intakesQueries[dateStr]?.data || [];
            const isToday = date.toDateString() === today.toDateString();
            const dayName = dayNames[date.getDay()];
            const takenCount = intakes.filter((i: any) => i.takenTime).length;
            const totalCount = intakes.length;

            return (
              <Card
                key={dateStr}
                className={`border-0 shadow-lg transition ${
                  isToday ? "ring-2 ring-blue-500 bg-blue-50" : ""
                }`}
              >
                <CardHeader className="pb-3">
                  <CardTitle className="text-sm">
                    <div className="text-center">
                      <p className="font-bold text-gray-900">{dayName}</p>
                      <p className="text-xs text-gray-600">
                        {date.getDate()}
                      </p>
                    </div>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  {intakesQueries[dateStr]?.isLoading ? (
                    <div className="text-center text-sm text-gray-500">
                      Carregando...
                    </div>
                  ) : totalCount > 0 ? (
                    <div className="space-y-2">
                      {intakes.map((intake: any) => {
                        const med = medicationsQuery.data?.find(
                          (m) => m.id === intake.medicationId
                        );
                        return (
                          <button
                            key={intake.id}
                            onClick={() => {
                              if (!intake.takenTime) {
                                markTakenMutation.mutate({
                                  intakeId: intake.id,
                                });
                              }
                            }}
                            className="w-full text-left p-2 rounded-lg border-l-4 bg-white hover:bg-gray-50 transition text-xs"
                            style={{
                              borderLeftColor: med?.color || "#3b82f6",
                            }}
                          >
                            <div className="flex items-center justify-between">
                              <div className="flex-1">
                                <p className="font-semibold text-gray-900 truncate">
                                  {med?.name}
                                </p>
                                <p className="text-gray-600">
                                  {String(
                                    new Date(intake.scheduledTime).getHours()
                                  ).padStart(2, "0")}
                                  :
                                  {String(
                                    new Date(intake.scheduledTime).getMinutes()
                                  ).padStart(2, "0")}
                                </p>
                              </div>
                              {intake.takenTime ? (
                                <CheckCircle2 className="w-5 h-5 text-green-600 flex-shrink-0" />
                              ) : (
                                <Circle className="w-5 h-5 text-gray-300 flex-shrink-0" />
                              )}
                            </div>
                          </button>
                        );
                      })}

                      {/* Progress */}
                      <div className="mt-3 pt-3 border-t">
                        <div className="flex items-center justify-between text-xs">
                          <span className="text-gray-600">Progresso</span>
                          <span className="font-semibold text-gray-900">
                            {takenCount}/{totalCount}
                          </span>
                        </div>
                        <div className="w-full bg-gray-200 rounded-full h-2 mt-1">
                          <div
                            className="bg-green-500 h-2 rounded-full transition-all"
                            style={{
                              width: `${(takenCount / totalCount) * 100}%`,
                            }}
                          />
                        </div>
                      </div>
                    </div>
                  ) : (
                    <div className="text-center text-xs text-gray-500">
                      Sem medicamentos
                    </div>
                  )}
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Summary */}
        <Card className="mt-8 border-0 shadow-lg bg-gradient-to-r from-blue-500 to-indigo-600 text-white">
          <CardHeader>
            <CardTitle>Resumo da Semana</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-3 gap-4">
              <div>
                <p className="text-blue-100 text-sm">Total de Medicamentos</p>
                <p className="text-3xl font-bold">
                  {days.reduce(
                    (sum, date) => {
                      const dateStr = date.toISOString().split("T")[0];
                      return sum + (intakesQueries[dateStr]?.data?.length || 0);
                    },
                    0
                  )}
                </p>
              </div>
              <div>
                <p className="text-blue-100 text-sm">Tomados</p>
                <p className="text-3xl font-bold">
                  {days.reduce(
                    (sum, date) => {
                      const dateStr = date.toISOString().split("T")[0];
                      const intakes = intakesQueries[dateStr]?.data || [];
                      return (
                        sum + intakes.filter((i: any) => i.takenTime).length
                      );
                    },
                    0
                  )}
                </p>
              </div>
              <div>
                <p className="text-blue-100 text-sm">Taxa de Aderência</p>
                <p className="text-3xl font-bold">
                  {(() => {
                    const total = days.reduce(
                      (sum, date) => {
                        const dateStr = date.toISOString().split("T")[0];
                        return sum + (intakesQueries[dateStr]?.data?.length || 0);
                      },
                      0
                    );
                    const taken = days.reduce(
                      (sum, date) => {
                        const dateStr = date.toISOString().split("T")[0];
                        const intakes = intakesQueries[dateStr]?.data || [];
                        return (
                          sum + intakes.filter((i: any) => i.takenTime).length
                        );
                      },
                      0
                    );
                    return total > 0 ? Math.round((taken / total) * 100) : 0;
                  })()}
                  %
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
