import { and, eq, gte, lte } from "drizzle-orm";
import { drizzle } from "drizzle-orm/mysql2";
import { InsertUser, users, medications, medicationSchedules, medicationIntakes, InsertMedicationIntake } from "../drizzle/schema";
import { ENV } from './_core/env';

let _db: ReturnType<typeof drizzle> | null = null;

// Lazily create the drizzle instance so local tooling can run without a DB.
export async function getDb() {
  if (!_db && process.env.DATABASE_URL) {
    try {
      _db = drizzle(process.env.DATABASE_URL);
    } catch (error) {
      console.warn("[Database] Failed to connect:", error);
      _db = null;
    }
  }
  return _db;
}

export async function upsertUser(user: InsertUser): Promise<void> {
  if (!user.openId) {
    throw new Error("User openId is required for upsert");
  }

  const db = await getDb();
  if (!db) {
    console.warn("[Database] Cannot upsert user: database not available");
    return;
  }

  try {
    const values: InsertUser = {
      openId: user.openId,
    };
    const updateSet: Record<string, unknown> = {};

    const textFields = ["name", "email", "loginMethod"] as const;
    type TextField = (typeof textFields)[number];

    const assignNullable = (field: TextField) => {
      const value = user[field];
      if (value === undefined) return;
      const normalized = value ?? null;
      values[field] = normalized;
      updateSet[field] = normalized;
    };

    textFields.forEach(assignNullable);

    if (user.lastSignedIn !== undefined) {
      values.lastSignedIn = user.lastSignedIn;
      updateSet.lastSignedIn = user.lastSignedIn;
    }
    if (user.role !== undefined) {
      values.role = user.role;
      updateSet.role = user.role;
    } else if (user.openId === ENV.ownerOpenId) {
      values.role = 'admin';
      updateSet.role = 'admin';
    }

    if (!values.lastSignedIn) {
      values.lastSignedIn = new Date();
    }

    if (Object.keys(updateSet).length === 0) {
      updateSet.lastSignedIn = new Date();
    }

    await db.insert(users).values(values).onDuplicateKeyUpdate({
      set: updateSet,
    });
  } catch (error) {
    console.error("[Database] Failed to upsert user:", error);
    throw error;
  }
}

export async function getUserByOpenId(openId: string) {
  const db = await getDb();
  if (!db) {
    console.warn("[Database] Cannot get user: database not available");
    return undefined;
  }

  const result = await db.select().from(users).where(eq(users.openId, openId)).limit(1);

  return result.length > 0 ? result[0] : undefined;
}

export async function getUserMedications(userId: number) {
  const db = await getDb();
  if (!db) return [];
  
  return db
    .select()
    .from(medications)
    .where(eq(medications.userId, userId))
    .orderBy(medications.name);
}

export async function getMedicationWithSchedules(medicationId: number) {
  const db = await getDb();
  if (!db) return null;
  
  const med = await db
    .select()
    .from(medications)
    .where(eq(medications.id, medicationId))
    .limit(1);
  
  if (med.length === 0) return null;
  
  const schedules = await db
    .select()
    .from(medicationSchedules)
    .where(eq(medicationSchedules.medicationId, medicationId))
    .orderBy(medicationSchedules.hour, medicationSchedules.minute);
  
  return { ...med[0], schedules };
}

export async function getTodaysMedicationIntakes(userId: number, date: Date) {
  const db = await getDb();
  if (!db) return [];
  
  const startOfDay = new Date(date);
  startOfDay.setHours(0, 0, 0, 0);
  const endOfDay = new Date(date);
  endOfDay.setHours(23, 59, 59, 999);
  
  return db
    .select()
    .from(medicationIntakes)
    .where(
      and(
        eq(medicationIntakes.userId, userId),
        gte(medicationIntakes.scheduledTime, startOfDay),
        lte(medicationIntakes.scheduledTime, endOfDay)
      )
    )
    .orderBy(medicationIntakes.scheduledTime);
}

export async function createMedicationIntake(intake: InsertMedicationIntake) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  const result = await db.insert(medicationIntakes).values(intake);
  return result;
}

export async function updateMedicationIntakeTaken(intakeId: number, takenTime: Date) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  return db
    .update(medicationIntakes)
    .set({ takenTime })
    .where(eq(medicationIntakes.id, intakeId));
}

// TODO: add more feature queries here as your schema grows.
