import { getDb } from "./db";
import { googleCalendarSync } from "../drizzle/schema";
import { eq } from "drizzle-orm";

const GOOGLE_OAUTH_CLIENT_ID = process.env.GOOGLE_OAUTH_CLIENT_ID || "";
const GOOGLE_OAUTH_CLIENT_SECRET = process.env.GOOGLE_OAUTH_CLIENT_SECRET || "";
const GOOGLE_OAUTH_REDIRECT_URI = process.env.GOOGLE_OAUTH_REDIRECT_URI || "";

export async function getGoogleCalendarAuthUrl(userId: number): Promise<string> {
  const state = Buffer.from(JSON.stringify({ userId, timestamp: Date.now() })).toString("base64");

  const params = new URLSearchParams({
    client_id: GOOGLE_OAUTH_CLIENT_ID,
    redirect_uri: GOOGLE_OAUTH_REDIRECT_URI,
    response_type: "code",
    scope: "https://www.googleapis.com/auth/calendar",
    state,
    access_type: "offline",
    prompt: "consent",
  });

  return `https://accounts.google.com/o/oauth2/v2/auth?${params.toString()}`;
}

export async function exchangeCodeForToken(code: string, userId: number) {
  try {
    const response = await fetch("https://oauth2.googleapis.com/token", {
      method: "POST",
      headers: { "Content-Type": "application/x-www-form-urlencoded" },
      body: new URLSearchParams({
        client_id: GOOGLE_OAUTH_CLIENT_ID,
        client_secret: GOOGLE_OAUTH_CLIENT_SECRET,
        code,
        grant_type: "authorization_code",
        redirect_uri: GOOGLE_OAUTH_REDIRECT_URI,
      }).toString(),
    });

    const data = await response.json();

    if (!response.ok) {
      throw new Error(data.error || "Failed to exchange code for token");
    }

    const db = await getDb();
    if (!db) throw new Error("Database not available");

    // Save or update sync record
    await db
      .insert(googleCalendarSync)
      .values({
        userId,
        accessToken: data.access_token,
        refreshToken: data.refresh_token || null,
        expiresAt: new Date(Date.now() + data.expires_in * 1000),
        isEnabled: 1,
      })
      .onDuplicateKeyUpdate({
        set: {
          accessToken: data.access_token,
          refreshToken: data.refresh_token || null,
          expiresAt: new Date(Date.now() + data.expires_in * 1000),
          isEnabled: 1,
        },
      });

    return { success: true };
  } catch (error) {
    console.error("Failed to exchange code for token:", error);
    throw error;
  }
}

export async function createCalendarEvent(
  userId: number,
  medicationName: string,
  scheduledTime: Date,
  dosage?: string
) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");

  const sync = await db
    .select()
    .from(googleCalendarSync)
    .where(eq(googleCalendarSync.userId, userId))
    .limit(1);

  if (sync.length === 0 || !sync[0].isEnabled) {
    return { success: false, error: "Google Calendar not connected" };
  }

  const accessToken = sync[0].accessToken;

  try {
    const event = {
      summary: `💊 ${medicationName}${dosage ? ` (${dosage})` : ""}`,
      description: `Medicamento agendado via MedSchedule Pro`,
      start: {
        dateTime: scheduledTime.toISOString(),
        timeZone: "America/Sao_Paulo",
      },
      end: {
        dateTime: new Date(scheduledTime.getTime() + 15 * 60000).toISOString(),
        timeZone: "America/Sao_Paulo",
      },
      reminders: {
        useDefault: false,
        overrides: [
          { method: "notification", minutes: 5 },
          { method: "popup", minutes: 0 },
        ],
      },
    };

    const response = await fetch("https://www.googleapis.com/calendar/v3/calendars/primary/events", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${accessToken}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify(event),
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error?.message || "Failed to create calendar event");
    }

    const createdEvent = await response.json();
    return { success: true, eventId: createdEvent.id };
  } catch (error) {
    console.error("Failed to create calendar event:", error);
    throw error;
  }
}

export async function disconnectGoogleCalendar(userId: number) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");

  await db
    .update(googleCalendarSync)
    .set({ isEnabled: 0 })
    .where(eq(googleCalendarSync.userId, userId));

  return { success: true };
}

export async function getGoogleCalendarStatus(userId: number) {
  const db = await getDb();
  if (!db) return { connected: false };

  const sync = await db
    .select()
    .from(googleCalendarSync)
    .where(eq(googleCalendarSync.userId, userId))
    .limit(1);

  if (sync.length === 0) {
    return { connected: false };
  }

  return {
    connected: sync[0].isEnabled === 1,
    calendarId: sync[0].calendarId,
  };
}
