import { COOKIE_NAME } from "@shared/const";
import { z } from "zod";
import { getSessionCookieOptions } from "./_core/cookies";
import { systemRouter } from "./_core/systemRouter";
import { publicProcedure, protectedProcedure, router } from "./_core/trpc";
import {
  getUserMedications,
  getMedicationWithSchedules,
  getTodaysMedicationIntakes,
  updateMedicationIntakeTaken,
  getDb,
} from "./db";

export const appRouter = router({
  system: systemRouter,
  auth: router({
    me: publicProcedure.query((opts) => opts.ctx.user),
    logout: publicProcedure.mutation(({ ctx }) => {
      const cookieOptions = getSessionCookieOptions(ctx.req);
      ctx.res.clearCookie(COOKIE_NAME, { ...cookieOptions, maxAge: -1 });
      return {
        success: true,
      } as const;
    }),
  }),

  medications: router({
    list: protectedProcedure.query(({ ctx }) => {
      return getUserMedications(ctx.user.id);
    }),

    create: protectedProcedure
      .input(
        z.object({
          name: z.string().min(1),
          description: z.string().optional(),
          dosage: z.string().optional(),
          frequency: z.number().int().min(1),
          startDate: z.date(),
          endDate: z.date().optional(),
          color: z.string().default("#3b82f6"),
          notes: z.string().optional(),
          schedules: z.array(
            z.object({
              hour: z.number().int().min(0).max(23),
              minute: z.number().int().min(0).max(59),
              dayOfWeek: z.number().int().min(0).max(6).optional(),
            })
          ),
        })
      )
      .mutation(async ({ ctx, input }) => {
        const db = await getDb();
        if (!db) throw new Error("Database not available");

        const { medications, medicationSchedules } = await import(
          "../drizzle/schema"
        );

        try {
          // Insert medication and get the result
          const insertResult = await db.insert(medications).values({
            userId: ctx.user.id,
            name: input.name,
            description: input.description,
            dosage: input.dosage,
            frequency: input.frequency,
            startDate: input.startDate,
            endDate: input.endDate,
            color: input.color,
            notes: input.notes,
          });

          // Extract the medication ID from the insert result
          // Drizzle returns an object with insertId property for MySQL
          const medId = (insertResult as any)[0]?.id || (insertResult as any).insertId;

          if (!medId) {
            throw new Error("Failed to get medication ID from insert result");
          }

          // Insert schedules if provided
          if (input.schedules && input.schedules.length > 0) {
            const schedulesToInsert = input.schedules.map((s) => ({
              medicationId: medId,
              hour: s.hour,
              minute: s.minute,
              dayOfWeek: s.dayOfWeek || null,
            }));

            await db.insert(medicationSchedules).values(schedulesToInsert);
          }

          return { id: medId, success: true };
        } catch (error) {
          console.error("Failed to create medication:", error);
          throw error;
        }
      }),

    getWithSchedules: protectedProcedure
      .input(z.object({ id: z.number() }))
      .query(({ input }) => {
        return getMedicationWithSchedules(input.id);
      }),
  }),

  intakes: router({
    today: protectedProcedure
      .input(z.object({ date: z.date() }))
      .query(({ ctx, input }) => {
        return getTodaysMedicationIntakes(ctx.user.id, input.date);
      }),

    markTaken: protectedProcedure
      .input(z.object({ intakeId: z.number() }))
      .mutation(({ input }) => {
        return updateMedicationIntakeTaken(input.intakeId, new Date());
      }),
  }),

  googleCalendar: router({
    getStatus: protectedProcedure.query(async ({ ctx }) => {
      const { getGoogleCalendarStatus } = await import("./google-calendar");
      return getGoogleCalendarStatus(ctx.user.id);
    }),
  }),
});

export type AppRouter = typeof appRouter;
