import { int, mysqlEnum, mysqlTable, text, timestamp, varchar } from "drizzle-orm/mysql-core";

/**
 * Core user table backing auth flow.
 * Extend this file with additional tables as your product grows.
 * Columns use camelCase to match both database fields and generated types.
 */
export const users = mysqlTable("users", {
  /**
   * Surrogate primary key. Auto-incremented numeric value managed by the database.
   * Use this for relations between tables.
   */
  id: int("id").autoincrement().primaryKey(),
  /** Manus OAuth identifier (openId) returned from the OAuth callback. Unique per user. */
  openId: varchar("openId", { length: 64 }).notNull().unique(),
  name: text("name"),
  email: varchar("email", { length: 320 }),
  loginMethod: varchar("loginMethod", { length: 64 }),
  role: mysqlEnum("role", ["user", "admin"]).default("user").notNull(),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
  lastSignedIn: timestamp("lastSignedIn").defaultNow().notNull(),
});

export type User = typeof users.$inferSelect;
export type InsertUser = typeof users.$inferInsert;

export const medications = mysqlTable("medications", {
  id: int("id").autoincrement().primaryKey(),
  userId: int("userId").notNull().references(() => users.id, { onDelete: "cascade" }),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  dosage: varchar("dosage", { length: 100 }), // e.g., "500mg", "2 tablets"
  frequency: int("frequency").notNull(), // times per day
  startDate: timestamp("startDate").notNull(),
  endDate: timestamp("endDate"), // null for continuous medications
  isActive: int("isActive").default(1).notNull(), // 0 or 1 for boolean
  color: varchar("color", { length: 7 }).default("#3b82f6"), // hex color for UI
  notes: text("notes"),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
});

export type Medication = typeof medications.$inferSelect;
export type InsertMedication = typeof medications.$inferInsert;

export const medicationSchedules = mysqlTable("medicationSchedules", {
  id: int("id").autoincrement().primaryKey(),
  medicationId: int("medicationId").notNull().references(() => medications.id, { onDelete: "cascade" }),
  hour: int("hour").notNull(), // 0-23
  minute: int("minute").notNull(), // 0-59
  dayOfWeek: int("dayOfWeek"), // 0-6 (Sunday-Saturday), null for daily
  createdAt: timestamp("createdAt").defaultNow().notNull(),
});

export type MedicationSchedule = typeof medicationSchedules.$inferSelect;
export type InsertMedicationSchedule = typeof medicationSchedules.$inferInsert;

export const medicationIntakes = mysqlTable("medicationIntakes", {
  id: int("id").autoincrement().primaryKey(),
  medicationId: int("medicationId").notNull().references(() => medications.id, { onDelete: "cascade" }),
  userId: int("userId").notNull().references(() => users.id, { onDelete: "cascade" }),
  scheduledTime: timestamp("scheduledTime").notNull(),
  takenTime: timestamp("takenTime"), // null if not taken
  notes: text("notes"),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
});

export type MedicationIntake = typeof medicationIntakes.$inferSelect;
export type InsertMedicationIntake = typeof medicationIntakes.$inferInsert;

export const googleCalendarSync = mysqlTable("googleCalendarSync", {
  id: int("id").autoincrement().primaryKey(),
  userId: int("userId").notNull().references(() => users.id, { onDelete: "cascade" }),
  accessToken: text("accessToken").notNull(),
  refreshToken: text("refreshToken"),
  expiresAt: timestamp("expiresAt"),
  calendarId: varchar("calendarId", { length: 255 }),
  isEnabled: int("isEnabled").default(1).notNull(),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
});

export type GoogleCalendarSync = typeof googleCalendarSync.$inferSelect;
export type InsertGoogleCalendarSync = typeof googleCalendarSync.$inferInsert;