const CACHE_NAME = "med-schedule-v1";
const urlsToCache = [
  "/",
  "/index.html",
  "/manifest.json",
];

// Install event
self.addEventListener("install", (event) => {
  event.waitUntil(
    caches.open(CACHE_NAME).then((cache) => {
      return cache.addAll(urlsToCache);
    })
  );
  self.skipWaiting();
});

// Activate event
self.addEventListener("activate", (event) => {
  event.waitUntil(
    caches.keys().then((cacheNames) => {
      return Promise.all(
        cacheNames.map((cacheName) => {
          if (cacheName !== CACHE_NAME) {
            return caches.delete(cacheName);
          }
        })
      );
    })
  );
  self.clients.claim();
});

// Fetch event - Network first, fallback to cache
self.addEventListener("fetch", (event) => {
  // Skip non-GET requests
  if (event.request.method !== "GET") {
    return;
  }

  // Skip API calls - let them fail gracefully
  if (event.request.url.includes("/api/")) {
    return;
  }

  event.respondWith(
    fetch(event.request)
      .then((response) => {
        // Cache successful responses
        if (response.status === 200) {
          const responseToCache = response.clone();
          caches.open(CACHE_NAME).then((cache) => {
            cache.put(event.request, responseToCache);
          });
        }
        return response;
      })
      .catch(() => {
        // Fallback to cache
        return caches.match(event.request).then((response) => {
          return response || new Response("Offline - Resource not available", {
            status: 503,
            statusText: "Service Unavailable",
          });
        });
      })
  );
});

// Background sync for medication intakes
self.addEventListener("sync", (event) => {
  if (event.tag === "sync-medication-intakes") {
    event.waitUntil(
      // Sync pending medication intakes when back online
      (async () => {
        try {
          const db = await openIndexedDB();
          const pendingIntakes = await getPendingIntakes(db);
          
          for (const intake of pendingIntakes) {
            try {
              await fetch("/api/trpc/intakes.markTaken", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ intakeId: intake.id }),
              });
              await markIntakeSynced(db, intake.id);
            } catch (error) {
              console.error("Failed to sync intake:", error);
            }
          }
        } catch (error) {
          console.error("Sync failed:", error);
        }
      })()
    );
  }
});

// Helper functions for IndexedDB
function openIndexedDB() {
  return new Promise((resolve, reject) => {
    const request = indexedDB.open("MedScheduleDB", 1);
    request.onerror = () => reject(request.error);
    request.onsuccess = () => resolve(request.result);
    request.onupgradeneeded = (event) => {
      const db = event.target.result;
      if (!db.objectStoreNames.contains("pendingIntakes")) {
        db.createObjectStore("pendingIntakes", { keyPath: "id" });
      }
    };
  });
}

function getPendingIntakes(db) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction(["pendingIntakes"], "readonly");
    const store = transaction.objectStore("pendingIntakes");
    const request = store.getAll();
    request.onerror = () => reject(request.error);
    request.onsuccess = () => resolve(request.result);
  });
}

function markIntakeSynced(db, intakeId) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction(["pendingIntakes"], "readwrite");
    const store = transaction.objectStore("pendingIntakes");
    const request = store.delete(intakeId);
    request.onerror = () => reject(request.error);
    request.onsuccess = () => resolve();
  });
}
