import { useEffect, useRef, useState } from "react";

export interface AlarmSettings {
  enabled: boolean;
  soundEnabled: boolean;
  vibrationEnabled: boolean;
  minutesBefore: number; // Notificar X minutos antes
}

export function useAdvancedAlarms(defaultSettings?: Partial<AlarmSettings>) {
  const [settings, setSettings] = useState<AlarmSettings>({
    enabled: true,
    soundEnabled: true,
    vibrationEnabled: true,
    minutesBefore: 5,
    ...defaultSettings,
  });

  const audioContextRef = useRef<AudioContext | null>(null);
  const oscillatorRef = useRef<OscillatorNode | null>(null);

  // Initialize audio context
  useEffect(() => {
    const initAudioContext = () => {
      if (!audioContextRef.current) {
        const audioContext = new (window.AudioContext || (window as any).webkitAudioContext)();
        audioContextRef.current = audioContext;
      }
    };

    // Initialize on user interaction
    document.addEventListener("click", initAudioContext, { once: true });
    return () => document.removeEventListener("click", initAudioContext);
  }, []);

  const playAlarmSound = (duration: number = 3000) => {
    if (!settings.soundEnabled || !audioContextRef.current) return;

    try {
      const audioContext = audioContextRef.current;
      const now = audioContext.currentTime;

      // Create oscillator for alarm sound
      const oscillator = audioContext.createOscillator();
      const gainNode = audioContext.createGain();

      oscillator.connect(gainNode);
      gainNode.connect(audioContext.destination);

      // Set frequency pattern for alarm sound
      oscillator.frequency.setValueAtTime(800, now);
      oscillator.frequency.setValueAtTime(600, now + 0.2);

      // Set volume
      gainNode.gain.setValueAtTime(0.3, now);
      gainNode.gain.exponentialRampToValueAtTime(0.01, now + duration / 1000);

      oscillator.start(now);
      oscillator.stop(now + duration / 1000);

      oscillatorRef.current = oscillator;
    } catch (error) {
      console.error("Failed to play alarm sound:", error);
    }
  };

  const triggerVibration = () => {
    if (!settings.vibrationEnabled) return;

    if ("vibrate" in navigator) {
      // Pattern: vibrate for 200ms, pause 100ms, vibrate for 200ms
      navigator.vibrate([200, 100, 200]);
    }
  };

  const stopAlarm = () => {
    if (oscillatorRef.current) {
      try {
        oscillatorRef.current.stop();
      } catch (error) {
        console.error("Failed to stop alarm:", error);
      }
      oscillatorRef.current = null;
    }
  };

  const triggerAlarm = () => {
    if (!settings.enabled) return;

    playAlarmSound(3000);
    triggerVibration();
  };

  const saveSettings = (newSettings: Partial<AlarmSettings>) => {
    const updated = { ...settings, ...newSettings };
    setSettings(updated);
    localStorage.setItem("alarmSettings", JSON.stringify(updated));
  };

  // Load settings from localStorage on mount
  useEffect(() => {
    const saved = localStorage.getItem("alarmSettings");
    if (saved) {
      try {
        setSettings(JSON.parse(saved));
      } catch (error) {
        console.error("Failed to load alarm settings:", error);
      }
    }
  }, []);

  return {
    settings,
    saveSettings,
    triggerAlarm,
    stopAlarm,
    playAlarmSound,
    triggerVibration,
  };
}
