import { useEffect, useRef } from "react";
import { trpc } from "@/lib/trpc";

export function useMedicationNotifications() {
  const notificationRef = useRef<Notification | null>(null);
  const intakesQuery = trpc.intakes.today.useQuery(
    { date: new Date() },
    { refetchInterval: 60000 } // Refetch every minute
  );

  useEffect(() => {
    if (!("Notification" in window)) {
      console.log("Browser does not support notifications");
      return;
    }

    // Request notification permission if not already granted
    if (Notification.permission === "default") {
      Notification.requestPermission();
    }

    // Check for upcoming medications every minute
    const interval = setInterval(() => {
      const now = new Date();
      const currentHour = now.getHours();
      const currentMinute = now.getMinutes();

      intakesQuery.data?.forEach((intake) => {
        if (!intake.takenTime) {
          const scheduledTime = new Date(intake.scheduledTime);
          const scheduledHour = scheduledTime.getHours();
          const scheduledMinute = scheduledTime.getMinutes();

          // Notify 5 minutes before scheduled time
          if (
            currentHour === scheduledHour &&
            currentMinute === scheduledMinute - 5
          ) {
            showNotification("Medicamento em 5 minutos!");
          }

          // Notify at scheduled time
          if (
            currentHour === scheduledHour &&
            currentMinute === scheduledMinute
          ) {
            showNotification("Hora de tomar seu medicamento!");
          }
        }
      });
    }, 60000); // Check every minute

    return () => clearInterval(interval);
  }, [intakesQuery.data]);

  const showNotification = (message: string) => {
    if (Notification.permission === "granted") {
      notificationRef.current = new Notification("MedSchedule Pro", {
        body: message,
        icon: "/pill-icon.svg",
        tag: "medication-reminder",
        requireInteraction: true,
      });

      // Auto-close after 10 seconds
      setTimeout(() => {
        notificationRef.current?.close();
      }, 10000);
    }
  };

  return { intakesQuery };
}
